#include "main.h"

#define INST MyFrame::instance


DEFINE_EVENT_TYPE(EVT_STATUS)
//------------------------------------------------------------------------------
IMPLEMENT_APP(MyApp)
MyFrame *MyFrame::instance = 0;
AnimationThread *MyApp::animationThread = 0;
//------------------------------------------------------------------------------

bool MyApp::OnInit(){
  MyFrame *frame = new MyFrame( _T(APPNAME), wxDefaultPosition, wxSize(800,700) );
  frame->CentreOnScreen();
  MyFrame::instance = frame;
  frame->Show(true);
  SetTopWindow(frame);
  
  //start thread
  animationThread = new AnimationThread();
  if(animationThread->Create()!=wxTHREAD_NO_ERROR){
    wxMessageBox(_T("Error creating animation thread."));
  }
  else animationThread->Run();
  
  return true;
}

//------------------------------------------------------------------------------
void MyFrame::setImgColor(wxImage& img1,wxColour c){
  unsigned char *pix1 = (unsigned char*)malloc(30*20*3);
  for(int i=0;i<30*20;i++){
    pix1[i*3  ] = c.Red();
    pix1[i*3+1] = c.Green();
    pix1[i*3+2] = c.Blue();
  }
  img1.SetData(pix1);
}


//------------------------------------------------------------------------------

MyFrame::MyFrame(const wxString& title, const wxPoint& pos, const wxSize& size)
								:wxFrame((wxFrame *)NULL, -1, title, pos, size) {
  
  ticks = 0;
  mouseX = 0;
  mouseY = 0;
  mouseIsDown = false;
  viewRotation.y = 70;
  
  CreateStatusBar();
  
#if defined(TARGET_WIN32)
	SetIcon(wxIcon(wxICON(IDI_MAIN)));
#endif

  //set up accelerator table
  wxAcceleratorEntry entries[2];
  entries[0].Set(wxACCEL_CMD,(int)'F',ID_Fullscreen);
  entries[1].Set(wxACCEL_CMD,(int)'S',wxID_SAVE);
  entries[1].Set(wxACCEL_CMD,(int)'O',wxID_OPEN);
  wxAcceleratorTable accel(2, entries);
  SetAcceleratorTable(accel);
  
  wxMenu *menuFile = new wxMenu;
  wxMenu *menuHelp = new wxMenu;
  wxMenu *menuView = new wxMenu;
  
  menuFile->Append( wxID_OPEN, _T("&Open\tCTRL-O") );
  menuFile->Append( wxID_SAVE, _T("&Save As...\tCTRL-S") );
  menuFile->AppendSeparator();
  menuFile->Append( ID_SaveImage, _T("&Export Image") );
  menuFile->Append( ID_SavePDF, _T("&Export PDF") );
  menuFile->Append( ID_SaveSVG, _T("&Export SVG") );
  menuFile->Append( ID_SavePointList, _T("&Export 3D Points") );
  menuFile->Append( wxID_EXIT, _T("E&xit") );
  menuHelp->Append( wxID_ABOUT, _T("&About...") );
  menuView->AppendCheckItem( ID_Fullscreen, _T("&Fullscreen\tCTRL-F") );
  
  wxMenuBar *menuBar = new wxMenuBar;
  menuBar->Append( menuFile, _T("&File") );
  menuBar->Append( menuView, _T("&View") );
  menuBar->Append( menuHelp, _T("&Help") );
  SetMenuBar( menuBar );
  
  auiManager = new wxAuiManager(this);
  
  //create opengl pane
  wxAuiPaneInfo paneInfo;
  paneInfo.CentrePane();
  paneInfo.PaneBorder(false);
  int attr[] = {WX_GL_RGBA,0,WX_GL_DOUBLEBUFFER,2};
  gl = new MyGLCanvas(this,ID_GL,wxDefaultPosition,wxDefaultSize,0,_T("OpenGLCanvas1"),attr);
  auiManager->AddPane(gl,paneInfo);
  
  //add sliders panel
  panel1 = new wxPanel(this);
  wxFlexGridSizer *panelSizer = new wxFlexGridSizer(2);
  panelSizer->AddGrowableCol(1);
  panel1->SetSizer(panelSizer);
  panel1->SetAutoLayout(true);
  
  sliders.push_back(new wxSlider(panel1,ID_Slider0,400,0,2000));
  panelSizer->Add(new wxStaticText(panel1,wxID_ANY,_T("Radius"),
          wxDefaultPosition,wxSize(50,20),wxALIGN_RIGHT|wxST_NO_AUTORESIZE),0,wxALL,MARGIN);
  panelSizer->Add(sliders.back(),0,wxEXPAND|wxALL,MARGIN);

  sliders.push_back(new wxSlider(panel1,ID_Slider1,500,0,2000));  
  panelSizer->Add(new wxStaticText(panel1,wxID_ANY,_T("Depth"),
	  wxDefaultPosition,wxSize(50,20),wxALIGN_RIGHT|wxST_NO_AUTORESIZE),0,wxALL,MARGIN);
  panelSizer->Add(sliders.back(),0,wxEXPAND|wxALL,MARGIN);
  
  sliders.push_back(new wxSlider(panel1,ID_Slider2,200,1,1000));
  panelSizer->Add(new wxStaticText(panel1,wxID_ANY,wxString::FromUTF8("Degree"),
	  wxDefaultPosition,wxSize(50,20),wxALIGN_RIGHT|wxST_NO_AUTORESIZE),0,wxALL,MARGIN);
  panelSizer->Add(sliders.back(),0,wxEXPAND|wxALL,MARGIN);
  
  sliders.push_back(new wxSlider(panel1,ID_Slider3,200,0,3000));
  panelSizer->Add(new wxStaticText(panel1,wxID_ANY,wxString::FromUTF8("Spin"),
	  wxDefaultPosition,wxSize(50,20),wxALIGN_RIGHT|wxST_NO_AUTORESIZE),0,wxALL,MARGIN);
  panelSizer->Add(sliders.back(),0,wxEXPAND|wxALL,MARGIN);

  sliders.push_back(new wxSlider(panel1,ID_Slider4,10,0,3000));
  panelSizer->Add(new wxStaticText(panel1,wxID_ANY,wxString::FromUTF8("Length"),
	  wxDefaultPosition,wxSize(50,20),wxALIGN_RIGHT|wxST_NO_AUTORESIZE),0,wxALL,MARGIN);
  panelSizer->Add(sliders.back(),0,wxEXPAND|wxALL,MARGIN);

  sliders.push_back(new wxSlider(panel1,ID_Slider5,0,0,1000));
  panelSizer->Add(new wxStaticText(panel1,wxID_ANY,wxString::FromUTF8("Chaos"),
	  wxDefaultPosition,wxSize(50,20),wxALIGN_RIGHT|wxST_NO_AUTORESIZE),0,wxALL,MARGIN);
  panelSizer->Add(sliders.back(),0,wxEXPAND|wxALL,MARGIN);

  sliders.push_back(new wxSlider(panel1,ID_Slider6,0,0,1000));
  panelSizer->Add(new wxStaticText(panel1,wxID_ANY,wxString::FromUTF8("Seed"),
	  wxDefaultPosition,wxSize(50,20),wxALIGN_RIGHT|wxST_NO_AUTORESIZE),0,wxALL,MARGIN);
  panelSizer->Add(sliders.back(),0,wxEXPAND|wxALL,MARGIN);
  
  sliders.push_back(new wxSlider(panel1,ID_Slider7,2,0,500));
  panelSizer->Add(new wxStaticText(panel1,wxID_ANY,wxString::FromUTF8("Speed"),
	  wxDefaultPosition,wxSize(50,20),wxALIGN_RIGHT|wxST_NO_AUTORESIZE),0,wxALL,MARGIN);
  panelSizer->Add(sliders.back(),0,wxEXPAND|wxALL,MARGIN);
  
  
  //add sliders as aui pane
  paneInfo.Left();
  paneInfo.Caption(wxString::FromUTF8("Parameters"));
  paneInfo.CaptionVisible(true);
  paneInfo.PaneBorder(true);
  paneInfo.CloseButton(true);
  paneInfo.Floatable(true);
  paneInfo.Dockable(true);
  paneInfo.MinSize(wxSize(200,sliders.size()*35));
  paneInfo.FloatingSize(wxSize(400,sliders.size()*35));
  auiManager->AddPane(panel1, paneInfo  );
  panel1->Layout();
  
  
  //make a rendering sheet
  panel2 = new wxPanel(this);
  wxFlexGridSizer *panelSizer2 = new wxFlexGridSizer(2);
  panelSizer2->AddGrowableCol(1);
  panel2->SetSizer(panelSizer2);
  panel2->SetAutoLayout(true);
  
  panelSizer2->Add(new wxStaticText(panel2,wxID_ANY,wxString::FromUTF8("Line Smooth"),
				    wxDefaultPosition,wxSize(100,20),wxALIGN_RIGHT|wxST_NO_AUTORESIZE),
		   0,wxALIGN_CENTRE_VERTICAL|wxALL,MARGIN);
  checkBox1 = new wxCheckBox(panel2,wxID_ANY,_T(""));
  checkBox1->SetValue(true);
  panelSizer2->Add(checkBox1,0,wxALL,MARGIN);

  panelSizer2->Add(new wxStaticText(panel2,wxID_ANY,wxString::FromUTF8("Line Width"),
				    wxDefaultPosition,wxSize(100,20),wxALIGN_RIGHT|wxST_NO_AUTORESIZE),
		   0,wxALIGN_CENTRE_VERTICAL|wxALL,MARGIN);
  lineSpin = new wxSpinCtrl(panel2,wxID_ANY,_T("1"),wxDefaultPosition,wxDefaultSize,wxSP_ARROW_KEYS,1,100,1);
  panelSizer2->Add(lineSpin,0,wxALL|wxEXPAND,MARGIN);
  
  //prepare color
  wxImage img1(30,20);
  setImgColor(img1,*wxBLACK);
  panelSizer2->Add(new wxStaticText(panel2,wxID_ANY,wxString::FromUTF8("Background"),
				    wxDefaultPosition,wxSize(100,20),wxALIGN_RIGHT|wxST_NO_AUTORESIZE),
		   0,wxALIGN_CENTRE_VERTICAL|wxALL,MARGIN);
  bgButton = new wxBitmapButton(panel2,ID_BGColor,wxBitmap(img1),wxDefaultPosition,wxSize(30,20) );
  panelSizer2->Add(bgButton,0,wxALL,MARGIN);
  
  wxImage img2(30,20);
  setImgColor(img2,*wxWHITE);
  panelSizer2->Add(new wxStaticText(panel2,wxID_ANY,wxString::FromUTF8("Foreground"),
				    wxDefaultPosition,wxSize(100,20),wxALIGN_RIGHT|wxST_NO_AUTORESIZE),
		   0,wxALIGN_CENTRE_VERTICAL|wxALL,MARGIN);
  fgButton = new wxBitmapButton(panel2,ID_FGColor,wxBitmap(img2),wxDefaultPosition,wxSize(30,20) );
  panelSizer2->Add(fgButton,0,wxALL,MARGIN);
  

  panelSizer2->Add(new wxStaticText(panel2,wxID_ANY,wxString::FromUTF8("Opacity"),
				    wxDefaultPosition,wxSize(100,20),wxALIGN_RIGHT|wxST_NO_AUTORESIZE),
		   0,wxALIGN_CENTRE_VERTICAL|wxALL,MARGIN);
  alphaSpin = new wxSpinCtrl(panel2,wxID_ANY,_T("200"),wxDefaultPosition,wxDefaultSize,wxSP_ARROW_KEYS,0,255,200);
  panelSizer2->Add(alphaSpin,0,wxALL|wxEXPAND,MARGIN);

  
  panelSizer2->Add(new wxStaticText(panel2,wxID_ANY,wxString::FromUTF8("Lense Angle"),
				    wxDefaultPosition,wxSize(100,20),wxALIGN_RIGHT|wxST_NO_AUTORESIZE),
		   0,wxALIGN_CENTRE_VERTICAL|wxALL,MARGIN);
  lenseSpin = new wxSpinCtrl(panel2,wxID_ANY,_T("90"),wxDefaultPosition,wxDefaultSize,wxSP_ARROW_KEYS,3,179,90);
  panelSizer2->Add(lenseSpin,0,wxALL|wxEXPAND,MARGIN);
  
  //add that pane
  paneInfo.Caption(wxString::FromUTF8("Styles"));
  paneInfo.Left();
  paneInfo.MinSize(wxSize(200,200));
  paneInfo.FloatingSize(wxSize(200,200));
  auiManager->AddPane(panel2, paneInfo  );
  
  auiManager->Update();

  //allocate the new path based on slider4's default value
  pathSize = sliders[4]->GetValue()*100;
  path = new JPoint[pathSize];
  perlinCache1 = new float[pathSize];
  perlinCache2 = new float[pathSize];
  generatePerlins();
  perlinSeedChanged = false;
  perlinChanged = false;
  
  setBGColor(0,0,0);
  fgColor = wxColour(255,255,255);
  
}
//------------------------------------------------------------------------------
void MyFrame::setBGColor(int r,int g,int b){
  bgColor = wxColour(r,g,b);
  gl->SetBackgroundColour(bgColor);
}
//------------------------------------------------------------------------------
void MyFrame::generatePerlins(){
  
  status(_T("Baking..."));
  
  for(int i=0;i<pathSize;i++){ 
    float pseed1 = sliders[6]->GetValue()*0.01+i*0.001;
    float pseed2 = sliders[6]->GetValue()*0.02+i*0.001;
    if(sliders[5]->GetValue()==0){
      perlinCache1[i] = 0;
      perlinCache2[i] = 0;
    }else{
      perlinCache1[i] = PerlinNoise::PerlinNoise_2D(pseed1,1000);
      perlinCache2[i] = PerlinNoise::PerlinNoise_2D(pseed2,2000);
    }
  }
  
  status(_T(""));

}
//------------------------------------------------------------------------------
AnimationThread::AnimationThread():wxThread(wxTHREAD_JOINABLE){
  
}
//------------------------------------------------------------------------------
ExitCode AnimationThread::Entry(){
  //set up the animation timer
  while(true){
    int w;
    int h;
    INST->gl->GetSize(&w,&h);
    INST->gl->SetCurrent();
    INST->update();
    
    INST->setupGLView(w,h);
    INST->draw(w,h);
    INST->gl->SwapBuffers();
    INST->ticks++;
    if(TestDestroy())Exit();
    Sleep(1000.0/60);
  }
  
}
//------------------------------------------------------------------------------
void MyFrame::setupGLView(int w,int h){
    glViewport(0,0,w,h);
    glMatrixMode(GL_PROJECTION);
    glLoadIdentity();
    gluPerspective(INST->lenseSpin->GetValue(),(float)w/h,0.001,10000);
    glMatrixMode(GL_MODELVIEW);
    glLoadIdentity();
}
//------------------------------------------------------------------------------
void MyFrame::OnQuit(wxCommandEvent& WXUNUSED(event)){
  Close(true);
}

//------------------------------------------------------------------------------

void MyFrame::OnAbout(wxCommandEvent& WXUNUSED(event)){
  wxAboutDialogInfo info;
  info.SetName(_(APPNAME));
  info.SetVersion(_("1.0.0"));
  info.SetDescription(_("A program for playing with cyclic spirals."));
  info.SetCopyright(_T("(C) jtnimoy 2009"));
  wxAboutBox(info);
}
//------------------------------------------------------------------------------

void MyFrame::OnFullscreen(wxCommandEvent& WXUNUSED(event)){
  
  wxAuiPaneInfo& pane1 = auiManager->GetPane(panel1);
  wxAuiPaneInfo& pane2 = auiManager->GetPane(panel2);

  if(IsFullScreen()){
    GetStatusBar()->Show();
    pane1.Show();
    pane2.Show();
  }else{
    GetStatusBar()->Hide();
    pane1.Hide();
    pane2.Hide();
  }
  
  ShowFullScreen(!IsFullScreen() );
  
  auiManager->Update();
}

//------------------------------------------------------------------------------
void MyFrame::update(){
  if(pathSizeChanged){
    delete [] path;
    delete [] perlinCache1;
    delete [] perlinCache2;
    pathSize = sliders[4]->GetValue()*100;
    path = new JPoint[pathSize];
    perlinCache1 = new float[pathSize];
    perlinCache2 = new float[pathSize];
    generatePerlins();
    pathSizeChanged = false;
  }
  
  if(perlinChanged){
    generatePerlins();
    perlinChanged = false;
  }

  if(perlinSeedChanged){
    generatePerlins();    
    perlinSeedChanged = false;
  }
  
  //populate the points based on time and sliders
  int timeAdvance = 400;//this is just to get the spiral to be more interesting on startup
  float speed = sliders[7]->GetValue();
  float time = ticks*speed+timeAdvance;
  float spin = sliders[3]->GetValue()*0.001;
  float theta = (ticks*speed+timeAdvance)*spin;
  float degree = (float)sliders[2]->GetValue()*0.0005;
  float chaos = sliders[5]->GetValue()*0.01;
  float lift = sliders[1]->GetValue();
  float radius = sliders[0]->GetValue()*0.1;
  
  for(int i=0;i<pathSize;i++){
    theta += degree;
    float mutate1 = 1 + perlinCache1[i] * chaos;
    float mutate2 = 1 + perlinCache2[i] * chaos;
    float r = pow(cos((time+i)*0.001)*radius,2) * mutate1;
    float x = r * cos(theta);
    float y = r * sin(theta);
    float z = cos((time+i)*0.001)*lift * mutate2;
    path[i] = JPoint(x,y,z);
  }
  
  if(mouseIsDown){
    viewRotation.x += mouseX - pmouseX;
    viewRotation.y += mouseY - pmouseY;
  }
  pmouseX = mouseX;
  pmouseY = mouseY;
  

}
//------------------------------------------------------------------------------
void MyFrame::draw(int width,int height){
  
  //setup view

  glClearColor(bgColor.Red()/255.0,bgColor.Green()/255.0,bgColor.Blue()/255.0,1);
  glClear(GL_COLOR_BUFFER_BIT|GL_DEPTH_BUFFER_BIT);
  glPushMatrix();
  glScalef(1.0/height,1.0/height,1.0/height);
  
  glTranslatef(0,0,-500);
  
  glEnable(GL_BLEND);  
  glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
  
  glRotatef(viewRotation.x,0,1,0);
  glRotatef(viewRotation.y,1,0,0);
  
  glColor4f(fgColor.Red()/255.0,fgColor.Green()/255.0,fgColor.Blue()/255.0,alphaSpin->GetValue()/255.0);

  if(checkBox1->IsChecked()){
    glEnable(GL_LINE_SMOOTH);
    glHint(GL_LINE_SMOOTH_HINT,GL_NICEST);
  }else{
    glDisable(GL_LINE_SMOOTH);
  }
  
  glLineWidth(lineSpin->GetValue());
  
  glBegin(GL_LINE_STRIP);
  for(int i=0;i<pathSize;i++)path[i].glVertex();
  glEnd();
  
  glPopMatrix();
  glFlush();
}

//------------------------------------------------------------------------------

void MyFrame::OnSize(wxSizeEvent& event){
  event.Skip();
}

//------------------------------------------------------------------------------
void MyGLCanvas::OnMouseCaptureLost(wxMouseCaptureLostEvent& event){
  INST->mouseIsDown = false;
}
//------------------------------------------------------------------------------
void MyGLCanvas::OnMousePressed(wxMouseEvent& event){
  CaptureMouse();
  wxPoint p = event.GetPosition();
  INST->mouseX = p.x;
  INST->mouseY = p.y;
  INST->mouseIsDown = true;  
  
  event.Skip();
}

//------------------------------------------------------------------------------

void MyGLCanvas::OnMouseReleased(wxMouseEvent& event){
  if(HasCapture())ReleaseMouse();
  wxPoint p = event.GetPosition();
  INST->mouseX = p.x;
  INST->mouseY = p.y;
  INST->mouseIsDown = false;
  
  event.Skip();  
}

//------------------------------------------------------------------------------

void MyGLCanvas::OnMouseMoved(wxMouseEvent& event){
  wxPoint p = event.GetPosition();
  INST->mouseX = p.x;
  INST->mouseY = p.y;
  
  event.Skip();
}

//------------------------------------------------------------------------------

MyGLCanvas::MyGLCanvas(wxWindow*parent,wxWindowID id,const wxPoint& pos,
					   const wxSize& size,long style,const wxString& name,int* attr):
						wxGLCanvas(parent,id,pos,size,style,name,attr){

}

//------------------------------------------------------------------------------
void MyFrame::OnClose(wxCloseEvent& event){
  MyApp::animationThread->Kill();
  exit(0);
}
//------------------------------------------------------------------------------
void MyFrame::OnSlider4ThumbTrack(wxScrollEvent& event){
  pathSizeChanged = true;
}
//------------------------------------------------------------------------------
void MyFrame::OnSlider5ThumbTrack(wxScrollEvent& event){
  perlinChanged = true;
}
//------------------------------------------------------------------------------
void MyFrame::OnSlider6ThumbTrack(wxScrollEvent& event){
  perlinSeedChanged = true;
}
//------------------------------------------------------------------------------
void MyFrame::OnSlider4Changed(wxScrollEvent& event){
  pathSizeChanged = true;
}
//------------------------------------------------------------------------------
void MyFrame::OnSlider5Changed(wxScrollEvent& event){
  perlinChanged = true;
}
//------------------------------------------------------------------------------
void MyFrame::OnSlider6Changed(wxScrollEvent& event){
  perlinSeedChanged = true;
}
//------------------------------------------------------------------------------
void MyFrame::OnBGColorPressed(wxCommandEvent& event){
  bgColor = wxGetColourFromUser(this,bgColor);
  setBGColor(bgColor.Red(),bgColor.Green(),bgColor.Blue());
  wxImage img = bgButton->GetBitmapLabel().ConvertToImage();
  setImgColor(img,bgColor);
  bgButton->SetBitmapLabel(wxBitmap(img));
}

//------------------------------------------------------------------------------
void MyFrame::OnFGColorPressed(wxCommandEvent& event){
  fgColor = wxGetColourFromUser(this,fgColor);
  wxImage img = fgButton->GetBitmapLabel().ConvertToImage();
  setImgColor(img,fgColor);
  fgButton->SetBitmapLabel(wxBitmap(img));
}

//------------------------------------------------------------------------------
void MyFrame::OnSaveImage(wxCommandEvent& event){
  MyApp::animationThread->Pause();
  
  //get a file name from user
  wxFileDialog *fileDialog = new wxFileDialog(this,_T("Save As..."),_T(""),_T(""),_T("BMP|*.bmp"),wxFD_SAVE);
  if(fileDialog->ShowModal()==wxID_CANCEL){
    MyApp::animationThread->Resume();
    return;
  }
  wxString filePath = fileDialog->GetPath();
  //checking for file extension
  if(!filePath.MakeLower().EndsWith(_T(".bmp")))filePath+=_T(".bmp");  
  
  SaveImageWindow *rw = new SaveImageWindow(this);
  int retval = rw->ShowModal();
  if(retval==wxID_CANCEL){
    MyApp::animationThread->Resume();
    return;
  }
  //otherwise, make a rendering window
  
  long w;
  long h;
  rw->textWidth->GetValue().ToLong(&w);
  rw->textHeight->GetValue().ToLong(&h);
  rw->Destroy();
  
  TileRenderWindow *trw = new TileRenderWindow(this,w,h,filePath,INST->lenseSpin->GetValue());
  trw->ShowModal();
  trw->Destroy();
  
  //start animation again
  MyApp::animationThread->Resume();
}
//------------------------------------------------------------------------------
SaveImageWindow::SaveImageWindow(wxWindow* parent):
  wxDialog(parent,wxID_ANY,_T("Save As Image"),wxDefaultPosition,wxSize(500,400),wxDEFAULT_DIALOG_STYLE,_T("SaveImageWindow")){

  wxFlexGridSizer *sizer = new wxFlexGridSizer(2);
  SetSizer(sizer);
  int w,h;
  INST->gl->GetSize(&w,&h);
  textWidth = new wxTextCtrl(this,wxID_ANY,wxString::Format(_T("%i"),w),
			     wxDefaultPosition,wxDefaultSize,0,wxTextValidator(wxFILTER_NUMERIC));
  sizer->Add(new wxStaticText(this,wxID_ANY,_T("Width"),
          wxDefaultPosition,wxSize(50,20),wxALIGN_RIGHT|wxST_NO_AUTORESIZE),0,wxALL,MARGIN);
  sizer->Add(textWidth,0,wxALL,MARGIN);
  
  textHeight = new wxTextCtrl(this,wxID_ANY,wxString::Format(_T("%i"),h),
			      wxDefaultPosition,wxDefaultSize,0,wxTextValidator(wxFILTER_NUMERIC));
  sizer->Add(new wxStaticText(this,wxID_ANY,_T("Height"),
          wxDefaultPosition,wxSize(50,20),wxALIGN_RIGHT|wxST_NO_AUTORESIZE),0,wxALL,MARGIN);
  sizer->Add(textHeight,0,wxALL,MARGIN);

  wxButton *b1 = new wxButton(this,wxID_OK,_T("OK"));
  b1->SetDefault();
  sizer->Add(b1,0,wxALL,MARGIN);
  
  wxButton *b2 = new wxButton(this,wxID_CANCEL,_T("Cancel"));
  sizer->Add(b2,0,wxALL,MARGIN);
  
  Fit();
  CentreOnParent();
  
}


//------------------------------------------------------------------------------
TileRenderWindow::TileRenderWindow(wxWindow* parent,int w,int h,wxString filepath,float fov):
  wxDialog(parent,wxID_ANY,_T("Rendering Image..."),wxDefaultPosition,wxSize(265,380),wxDEFAULT_DIALOG_STYLE,_T("TileRenderWindow")){

  width = w;
  height = h;
  filePath = filepath;
  fovy = fov;
  wxBoxSizer *sizer = new wxBoxSizer(wxVERTICAL);
  
  //gl window
  int attr[] = {WX_GL_RGBA,0,WX_GL_DOUBLEBUFFER,2};
  wxGLCanvas *renderGL = new wxGLCanvas(this,wxID_ANY,wxDefaultPosition,wxSize(256,256),0,_T("OpenGLCanvas2"),attr);
  renderGL->SetBackgroundColour(INST->bgColor);
  renderGL->SetCurrent();
  glPixelStorei(GL_PACK_ALIGNMENT, 1);
  INST->setupGLView(w,h);
  
  sizer->Add(renderGL,0,wxALL,MARGIN);
  
  //progress bar
  gauge = new wxGauge(this,wxID_ANY,1000,wxDefaultPosition,wxDefaultSize,wxGA_HORIZONTAL);
  sizer->Add(gauge,0,wxALL|wxEXPAND|wxALIGN_CENTRE,MARGIN);
  
  //progress text
  text = new wxStaticText(this,wxID_ANY,_T("0% Complete"),wxDefaultPosition,wxSize(260,40),wxALIGN_CENTRE);
  sizer->Add(text,0,wxALL|wxEXPAND,MARGIN);

  //cancel button
  wxButton *button1 = new wxButton(this,wxID_CANCEL,_T("Cancel"));
  sizer->Add(button1,0,wxALL|wxALIGN_CENTRE,MARGIN);
  
  SetSizer(sizer);
  
  Fit();
  CentreOnParent();
  
  //BEGIN RENDERING
  
  //allocate memory
  image = (GLubyte*)malloc(w * h * 3 * sizeof(GLubyte));
  if (!image) {
    printf("Malloc failed!\n");
    Close();
  }
  
  //start tile rendering
  tr = trNew();
  trTileSize(tr,256,256,0);
  trImageSize(tr, w,h);
  trImageBuffer(tr, GL_RGB, GL_UNSIGNED_BYTE, image);
  
  trPerspective(tr,INST->lenseSpin->GetValue(),(float)w/h,0.001,10000);
  
  tileCount = 0;
  timer = new wxTimer(this);
  timer->Start(1);
  doneTiling = false;
}
//------------------------------------------------------------------------------
/*  
//render the tiles
*/
//------------------------------------------------------------------------------
void TileRenderWindow::OnTimer(wxTimerEvent& event){
  
  if(doneTiling>0){//coming back to empty, then finish up.
    if(doneTiling>20){
      //make a wx image
      wxImage *img = new wxImage(width,height,image);
      //flip vertically
      wxImage img2 = img->Mirror(false);
      //save to file
      img2.SaveFile(filePath);
      
      delete img;
      
      timer->Stop();
      Close();
    }
    doneTiling++;
    return;
  }
  
  trBeginTile(tr);
  INST->draw(width,height);
  
  int wDiv = ceil(width/256.0);
  int hDiv = ceil(height/256.0);
  float total = wDiv*hDiv;
  gauge->SetValue(((tileCount)/total)*1000);
  text->SetLabel(wxString::Format(_T("%i%% Complete"),(int)((tileCount/total)*100)));
  
  tileCount++;
  if(!trEndTile(tr)){
    doneTiling = 1;
    gauge->SetValue(1000);
    text->SetLabel(_T("Writing File to Disk"));
  }

}
//------------------------------------------------------------------------------
void TileRenderWindow::OnClose(wxCloseEvent& event){
  //useless cuz it's a dialog
  event.Skip();
}
//------------------------------------------------------------------------------
TileRenderWindow::~TileRenderWindow(){
  if(timer->IsRunning())timer->Stop();
  trDelete(tr);
  free(image);
}
//------------------------------------------------------------------------------
void MyFrame::OnSave(wxCommandEvent& event){
  MyApp::animationThread->Pause();
  
  //get a file name from user
  wxFileDialog *fileDialog = new wxFileDialog(this,_T("Save As..."),_T(""),_T(""),_T("Pirouette File|*.pirouette"),wxFD_SAVE);
  if(fileDialog->ShowModal()==wxID_CANCEL){
    MyApp::animationThread->Resume();
    return;
  }
  wxString filePath = fileDialog->GetPath();
  //checking for file extension
  if(!filePath.MakeLower().EndsWith(_T(".pirouette")))filePath+=_T(".pirouette");
  
  //make new document
  wxXmlDocument *doc = new wxXmlDocument();
  
  //make root node
  wxXmlNode *root = new wxXmlNode(wxXML_ELEMENT_NODE,_T("pirouette")); 
  doc->SetRoot(root);
  
  wxXmlNode *n;

  n = new wxXmlNode(wxXML_ELEMENT_NODE,_T("datestamp"));
  n->AddProperty( _T("value"),  wxDateTime::Now().FormatISODate() + _T("?") + wxDateTime::Now().FormatISOTime()   );
  root->AddChild(n);
  
  for(int i=0;i<sliders.size();i++){
    n = new wxXmlNode(wxXML_ELEMENT_NODE,_T("slider"));
    n->AddProperty( _T("value"), wxString::Format(_T("%i"),sliders[i]->GetValue()));
    root->AddChild(n);
  }
  
  n = new wxXmlNode(wxXML_ELEMENT_NODE,_T("linesmooth"));
  n->AddProperty( _T("value"), wxString::Format(_T("%i"),checkBox1->GetValue()));
  root->AddChild(n);
  
  n = new wxXmlNode(wxXML_ELEMENT_NODE,_T("linewidth"));
  n->AddProperty( _T("value"), wxString::Format(_T("%i"),lineSpin->GetValue()));
  root->AddChild(n);
  
  n = new wxXmlNode(wxXML_ELEMENT_NODE,_T("background"));
  n->AddProperty( _T("value"), bgColor.GetAsString(wxC2S_HTML_SYNTAX));
  root->AddChild(n);
  
  
  n = new wxXmlNode(wxXML_ELEMENT_NODE,_T("foreground"));
  n->AddProperty( _T("value"), fgColor.GetAsString(wxC2S_HTML_SYNTAX));
  root->AddChild(n);
  
  n = new wxXmlNode(wxXML_ELEMENT_NODE,_T("opacity"));
  n->AddProperty( _T("value"), wxString::Format(_T("%i"),alphaSpin->GetValue()));
  root->AddChild(n);
  
  n = new wxXmlNode(wxXML_ELEMENT_NODE,_T("fovy"));
  n->AddProperty( _T("value"), wxString::Format(_T("%i"),lenseSpin->GetValue()));
  root->AddChild(n);
  
  n = new wxXmlNode(wxXML_ELEMENT_NODE,_T("rotation"));
  n->AddProperty( _T("x"), wxString::Format(_T("%f"),viewRotation.x));
  n->AddProperty( _T("y"), wxString::Format(_T("%f"),viewRotation.y));
  n->AddProperty( _T("z"), wxString::Format(_T("%f"),viewRotation.z));
  root->AddChild(n);
  
  
  //save to disk
  doc->Save(filePath);
  
  //clean up
  delete doc;

  //start animation again
  MyApp::animationThread->Resume();
}
//------------------------------------------------------------------------------
void MyFrame::OnOpen(wxCommandEvent& event){
  MyApp::animationThread->Pause();
 
  //get a file name from user
  wxFileDialog *fileDialog = new wxFileDialog(this,_T("Open File..."),_T(""),_T(""),_T("Pirouette File|*.pirouette"),wxFD_OPEN);
  if(fileDialog->ShowModal()==wxID_CANCEL){
    MyApp::animationThread->Resume();
    return;
  }
  wxString filePath = fileDialog->GetPath();
  wxXmlDocument *doc = new wxXmlDocument(filePath);
  
  wxXmlNode *child = doc->GetRoot()->GetChildren();
  int sliderCount = 0;
  while (child) {
    
    if (child->GetName() == _T("slider")) {
      wxString valStr;
      child->GetPropVal(_T("value"),&valStr);
      long valInt;
      valStr.ToLong(&valInt);
      sliders[sliderCount]->SetValue(valInt);
      sliderCount++;
    }else if (child->GetName() == _T("linesmooth")) {
      wxString valStr;
      child->GetPropVal(_T("value"),&valStr);
      long valInt;
      valStr.ToLong(&valInt);
      checkBox1->SetValue(valInt);
      
    }else if (child->GetName() == _T("linewidth")) {
      wxString valStr;
      child->GetPropVal(_T("value"),&valStr);
      long valInt;
      valStr.ToLong(&valInt);
      lineSpin->SetValue(valInt);
      
    }else if (child->GetName() == _T("background")) {
      wxString valStr;
      child->GetPropVal(_T("value"),&valStr);
      bgColor.Set(valStr);
      gl->SetBackgroundColour(bgColor);
      wxImage img = bgButton->GetBitmapLabel().ConvertToImage();
      setImgColor(img,bgColor);
      bgButton->SetBitmapLabel(wxBitmap(img));
      
    }else if (child->GetName() == _T("foreground")) {
      wxString valStr;
      child->GetPropVal(_T("value"),&valStr);
      fgColor.Set(valStr);
      wxImage img = fgButton->GetBitmapLabel().ConvertToImage();
      setImgColor(img,fgColor);
      fgButton->SetBitmapLabel(wxBitmap(img));
      
    }else if (child->GetName() == _T("opacity")) {
      wxString valStr;
      child->GetPropVal(_T("value"),&valStr);
      long valInt;
      valStr.ToLong(&valInt);
      alphaSpin->SetValue(valInt);
      
    }else if (child->GetName() == _T("fovy")) {
      wxString valStr;
      child->GetPropVal(_T("value"),&valStr);
      long valInt;
      valStr.ToLong(&valInt);
      lenseSpin->SetValue(valInt);
      
    }else if (child->GetName() == _T("rotation")) {
      wxString valStr;
      double d;
      child->GetPropVal(_T("x"),&valStr);
      valStr.ToDouble(&d);
      viewRotation.x = d;
      
      child->GetPropVal(_T("y"),&valStr);
      valStr.ToDouble(&d);
      viewRotation.y = d;
      
      child->GetPropVal(_T("z"),&valStr);
      valStr.ToDouble(&d);
      viewRotation.z = d;
    }
    //advance to next sibling
    child = child->GetNext();
  }
  
  //update lines and randoms just in case
  wxScrollEvent se;
  OnSlider4ThumbTrack(se);
  
  delete doc;
  
  //start animation again
  MyApp::animationThread->Resume();
}
//------------------------------------------------------------------------------
void MyFrame::OnStatus(wxCommandEvent& event){
  SetStatusText(event.GetString());
}
//------------------------------------------------------------------------------
/*
  thread safe status text setter
 */
void MyFrame::status(wxString txt){
  wxCommandEvent event(EVT_STATUS,GetId());
  event.SetEventObject(this);
  event.SetString(txt);
  AddPendingEvent(event);
}
//------------------------------------------------------------------------------
void MyFrame::OnSavePDF(wxCommandEvent& event){
  MyApp::animationThread->Pause();
  
  //get a file name from user
  wxFileDialog *fileDialog = new wxFileDialog(this,_T("Save As..."),_T(""),_T(""),_T("Portable Document File|*.pdf"),wxFD_SAVE);
  if(fileDialog->ShowModal()==wxID_CANCEL){
    MyApp::animationThread->Resume();
    return;
  }
  wxString filePath = fileDialog->GetPath();
  //checking for file extension
  if(!filePath.MakeLower().EndsWith(_T(".pdf")))filePath+=_T(".pdf");
  
  FILE *fp;
  int state = GL2PS_OVERFLOW, buffsize = 0;
  
  int w,h;
  gl->GetSize(&w,&h);

  fp = fopen((const char *)filePath.fn_str(), "wb");
  while(state == GL2PS_OVERFLOW){
    buffsize += 1024*1024;
    gl2psBeginPage(APPNAME,APPNAME, NULL, GL2PS_PDF, GL2PS_SIMPLE_SORT, 
		   GL2PS_DRAW_BACKGROUND | GL2PS_USE_CURRENT_VIEWPORT, 
		   GL_RGBA, 0, NULL, 0, 0, 0, buffsize, fp, "");
    setupGLView(w,h);
    draw(w,h);
    state = gl2psEndPage();
  }
  fclose(fp);
  
  MyApp::animationThread->Resume();
}
//------------------------------------------------------------------------------
void MyFrame::OnSaveSVG(wxCommandEvent& event){
  MyApp::animationThread->Pause();
  
  //get a file name from user
  wxFileDialog *fileDialog = new wxFileDialog(this,_T("Save As..."),_T(""),_T(""),_T("Scalable Vector Graphics|*.svg"),wxFD_SAVE);
  if(fileDialog->ShowModal()==wxID_CANCEL){
    MyApp::animationThread->Resume();
    return;
  }
  wxString filePath = fileDialog->GetPath();
  //checking for file extension
  if(!filePath.MakeLower().EndsWith(_T(".svg")))filePath+=_T(".svg");
  
  FILE *fp;
  int state = GL2PS_OVERFLOW, buffsize = 0;
  
  int w,h;
  gl->GetSize(&w,&h);

  fp = fopen((const char *)filePath.fn_str(), "wb");
  while(state == GL2PS_OVERFLOW){
    buffsize += 1024*1024;
    gl2psBeginPage(APPNAME,APPNAME, NULL, GL2PS_SVG, GL2PS_SIMPLE_SORT, 
		   GL2PS_DRAW_BACKGROUND | GL2PS_USE_CURRENT_VIEWPORT, 
		   GL_RGBA, 0, NULL, 0, 0, 0, buffsize, fp, "");
    setupGLView(w,h);
    draw(w,h);
    state = gl2psEndPage();
  }
  fclose(fp);
  
  MyApp::animationThread->Resume();
}
//------------------------------------------------------------------------------
void MyFrame::OnSavePointList(wxCommandEvent& event){
  MyApp::animationThread->Pause();
  
  //get a file name from user
  wxFileDialog *fileDialog = new wxFileDialog(this,_T("Save As..."),_T(""),_T(""),_T("Text File|*.txt"),wxFD_SAVE);
  if(fileDialog->ShowModal()==wxID_CANCEL){
    MyApp::animationThread->Resume();
    return;
  }
  wxString filePath = fileDialog->GetPath();
  //checking for file extension
  if(!filePath.MakeLower().EndsWith(_T(".txt")))filePath+=_T(".txt");
  
  ofstream outfile(filePath.fn_str());
  
  for(int i=0;i<pathSize;i++){
    outfile << path[i].x << ", " << path[i].y << ", " << path[i].z << endl;
  }
  
  outfile.close();
  
  MyApp::animationThread->Resume();
}
